using Microsoft.AspNetCore.Mvc;
using ProductMicroservice.Models;
using ProductMicroservice.Services;

namespace ProductMicroservice.Controllers
{
	[ApiController]
	[Route("api/[controller]")]
	public class CustomersController : ControllerBase
	{
		private readonly ICustomerService _service;

		public CustomersController(ICustomerService service)
		{
			_service = service;
		}

		[HttpGet]
		public async Task<ActionResult<IEnumerable<Customer>>> GetAll(CancellationToken cancellationToken)
		{
			var customers = await _service.GetAllAsync(cancellationToken);
			return Ok(customers);
		}

		[HttpGet("{id:int}")]
		public async Task<ActionResult<Customer>> GetById(int id, CancellationToken cancellationToken)
		{
			var customer = await _service.GetByIdAsync(id, cancellationToken);
			if (customer == null)
			{
				return NotFound();
			}
			return Ok(customer);
		}

		[HttpPost]
		public async Task<ActionResult<Customer>> Create([FromBody] Customer customer, CancellationToken cancellationToken)
		{
			if (!ModelState.IsValid)
			{
				return ValidationProblem(ModelState);
			}

			var created = await _service.CreateAsync(customer, cancellationToken);
			return CreatedAtAction(nameof(GetById), new { id = created.CustomerId }, created);
		}

		[HttpPut("{id:int}")]
		public async Task<IActionResult> Update(int id, [FromBody] Customer customer, CancellationToken cancellationToken)
		{
			if (customer.CustomerId != 0 && id != customer.CustomerId)
			{
				return BadRequest("ID in route and body must match");
			}

			var success = await _service.UpdateAsync(id, customer, cancellationToken);
			if (!success)
			{
				return NotFound();
			}
			return NoContent();
		}

		// Alternate PUT for clients sending only body with customerId
		[HttpPut]
		public async Task<IActionResult> Update([FromBody] Customer customer, CancellationToken cancellationToken)
		{
			if (customer.CustomerId == 0)
			{
				return BadRequest("customerId is required in body");
			}
			var success = await _service.UpdateAsync(customer.CustomerId, customer, cancellationToken);
			if (!success)
			{
				return NotFound();
			}
			return NoContent();
		}

		[HttpDelete("{id:int}")]
		public async Task<IActionResult> Delete(int id, CancellationToken cancellationToken)
		{
			var success = await _service.DeleteAsync(id, cancellationToken);
			if (!success)
			{
				return NotFound();
			}
			return NoContent();
		}
	}
} 