using Microsoft.EntityFrameworkCore;
using ProductMicroservice.Data;
using ProductMicroservice.Models;

namespace ProductMicroservice.Services
{
	public class CustomerService : ICustomerService
	{
		private readonly AppDbContext _dbContext;

		public CustomerService(AppDbContext dbContext)
		{
			_dbContext = dbContext;
		}

		public async Task<List<Customer>> GetAllAsync(CancellationToken cancellationToken = default)
		{
			return await _dbContext.Customers.AsNoTracking().ToListAsync(cancellationToken);
		}

		public async Task<Customer?> GetByIdAsync(int id, CancellationToken cancellationToken = default)
		{
			return await _dbContext.Customers.AsNoTracking().FirstOrDefaultAsync(c => c.CustomerId == id, cancellationToken);
		}

		public async Task<Customer> CreateAsync(Customer customer, CancellationToken cancellationToken = default)
		{
			_dbContext.Customers.Add(customer);
			await _dbContext.SaveChangesAsync(cancellationToken);
			return customer;
		}

		public async Task<bool> UpdateAsync(int id, Customer customer, CancellationToken cancellationToken = default)
		{
			var existing = await _dbContext.Customers.FirstOrDefaultAsync(c => c.CustomerId == id, cancellationToken);
			if (existing == null)
			{
				return false;
			}

			existing.StoreId = customer.StoreId;
			existing.FirstName = customer.FirstName;
			existing.LastName = customer.LastName;
			existing.Email = customer.Email;
			existing.AddressId = customer.AddressId;
			existing.ActiveBool = customer.ActiveBool;
			existing.CreateDate = customer.CreateDate;
			existing.LastUpdate = customer.LastUpdate;

			await _dbContext.SaveChangesAsync(cancellationToken);
			return true;
		}

		public async Task<bool> DeleteAsync(int id, CancellationToken cancellationToken = default)
		{
			var exists = await _dbContext.Customers.AsNoTracking().AnyAsync(c => c.CustomerId == id, cancellationToken);
			if (!exists)
			{
				return false;
			}

			await using var tx = await _dbContext.Database.BeginTransactionAsync(cancellationToken);

			// Delete dependents first, then the customer
			await _dbContext.Database.ExecuteSqlRawAsync(
				"DELETE FROM public.payment WHERE customer_id = {0}", id);
			await _dbContext.Database.ExecuteSqlRawAsync(
				"DELETE FROM public.rental WHERE customer_id = {0}", id);
			await _dbContext.Database.ExecuteSqlRawAsync(
				"DELETE FROM public.customer WHERE customer_id = {0}", id);

			await tx.CommitAsync(cancellationToken);
			return true;
		}
	}
} 